# -*- coding: utf-8 -*-

"""
/***************************************************************************
 CompGIS Dialog
                                 A QGIS plugin
 
                             -------------------
        begin                : 2020-06-16
        git sha              : $Format:%H$
        copyright            : (C) 2024 by IGN/SGM/TS
        email                : travaux.speciaux@ign.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 TODO
"""
from __future__ import absolute_import

import os

from PyQt5 import uic, QtGui
from PyQt5.QtGui import QIcon
from PyQt5.QtWidgets import QDialog
from qgis.core import Qgis, QgsProject, QgsVectorLayer

from ..CompGisCore.line_manager import to_segments, points_crossing_segment, pairs, cut_last_slash



FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'ante_sim.ui'))


class AntesimWidget(QDialog, FORM_CLASS):

    def __init__(self, plugin_dir, iface, parent=None):
        """Constructor."""
        super(AntesimWidget, self).__init__(parent)
        self.setWindowIcon(QIcon(os.path.join(plugin_dir,"resources/antesim.png")))
       
        self.setupUi(self)
        self.iface = iface
        self.plugin_dir = plugin_dir
        

    def connecting_signals(self):
        self.disconnecting_signals()
        self.bouton_nlle_sim.clicked.connect(self.new_simulation)
        self.bouton_generer_sim.clicked.connect(self.generate_simulation)
        
        self.populate_group_combobox()

    def disconnecting_signals(self):
        try:
            self.bouton_nlle_sim.clicked.disconnect()
        except Exception as e:
            pass
        try:
            self.bouton_creer_sim.clicked.disconnect()
        except Exception as e:
            pass
            
    def load_style_sim(self, lyr):
        """ Styles for Antesim """
        layer_type = lyr.name().split(' - ')[0]
        style_dir = os.path.join(self.plugin_dir,"default_style")
        if layer_type == 'Stations':
            path_style = os.path.join(style_dir,'style_stations.qml')
            lyr.loadNamedStyle(path_style)
        elif layer_type == 'Targets':
            path_style = os.path.join(style_dir, 'style_targets.qml')
            lyr.loadNamedStyle(path_style)
        elif layer_type == 'Obs z':
            path_style = os.path.join(style_dir, 'style_obs_z.qml')
            lyr.loadNamedStyle(path_style)
        elif layer_type == 'Obs 3D':
            path_style = os.path.join(style_dir, 'style_obs_3D.qml')
            lyr.loadNamedStyle(path_style)
        return lyr

    def create_4_layers_in_group(self,src,name):
        """ Create the 4 layers for a simulation
        scr in proj format
        """
        endUri = "?crs=PROJ:{}&field=name:string(20)&index=yes".format(src)
        uriPts = "Point"+endUri+"&field=fixed:bool&field=ellipsoid_height:double"
        uriLin = "Linestring"+endUri
        
        root = QgsProject.instance().layerTreeRoot()
        group = root.addGroup(name)
        
        stationPoints = QgsVectorLayer(uriPts, "Stations - {}".format(name), "memory")
        self.load_style_sim(stationPoints)
        targetPoints = QgsVectorLayer(uriPts, "Targets - {}".format(name), "memory")
        self.load_style_sim(targetPoints)
        obsZLines = QgsVectorLayer(uriLin, "Obs z - {}".format(name), "memory")
        self.load_style_sim(obsZLines)
        obs3DLines = QgsVectorLayer(uriLin, "Obs 3D - {}".format(name), "memory")
        self.load_style_sim(obs3DLines)
        
        QgsProject.instance().addMapLayer(stationPoints, False) 
        group.addLayer(stationPoints)
        QgsProject.instance().addMapLayer(targetPoints, False)  # False to avoid duplicata
        group.addLayer(targetPoints)
        QgsProject.instance().addMapLayer(obsZLines, False) 
        group.addLayer(obsZLines)
        QgsProject.instance().addMapLayer(obs3DLines, False)   
        group.addLayer(obs3DLines)
        
    def getPoint(self, feature):
        """ takes a point qgis feature and extract the info """
        pt = {}
        pt['x'] = feature.geometry().asPoint().x()
        pt['y'] = feature.geometry().asPoint().y()
        pt['z'] = feature.attribute('ellipsoid_height')
        pt['name'] = feature.attribute('name')
        return(pt)
        
    def findPoints(self, group):
        """ Find all the points that are in the Point layers of the group group.
        
        Default value of ellipsoid height is 100 ; points with no h_ell take 
        the h_ell of the last point which has h_ell, or the default value 
        (in the order in which you wrote them in QGIS ; stations then targets)
        """
        points = []
        fixed_points = []
        ell_height = 100
        root = QgsProject.instance().layerTreeRoot()
        sim_group = root.findGroup(group)
        for layer_tree_layer in sim_group.children():
            if 'Obs' not in layer_tree_layer.name():   #on est sur une couche de points
                for feature in layer_tree_layer.layer().getFeatures():
                    pt = self.getPoint(feature)
                    if pt['z'] == None:
                        pt['z'] = ell_height
                    else:
                        ell_height = pt['z']
                    if feature.attribute('fixed'):
                        fixed_points.append(pt)
                    else:
                        points.append(pt)
        return(fixed_points,points)    
        
    def findObs(self, group):
        """ Find all the observations that are in the obs layers of the group group.
        """
        niv_obs = []
        tD_obs = []
        
        # Get point layers
        station_lay = QgsProject.instance().mapLayersByName('Stations - ' + group)[0]
        target_lay = QgsProject.instance().mapLayersByName('Targets - ' + group)[0]
        # -- Get niv obs --
        niv_layer = QgsProject.instance().mapLayersByName('Obs z - ' + group)[0]
        for feature in niv_layer.getFeatures():
            segments = to_segments(feature)
            for segment in segments :
                dep_pts = points_crossing_segment(segment, station_lay)
                if len(dep_pts) > 0:
                    pts = points_crossing_segment(segment, target_lay)
                    niv_obs += pairs(dep_pts, pts)
        # -- Get 3D obs --
        obs_layer = QgsProject.instance().mapLayersByName('Obs 3D - ' + group)[0]
        for feature in obs_layer.getFeatures():
            segments = to_segments(feature)
            for segment in segments :
                dep_pts = points_crossing_segment(segment, station_lay)
                if len(dep_pts) > 0:
                    pts = points_crossing_segment(segment, target_lay)
                    tD_obs += pairs(dep_pts, pts) 
        return(niv_obs,tD_obs)
        
    def create_cor_file(self, group, chemin_comp, sigX, sigY, sigZ):
        """ Create a .cor file with group as name and with path of chemin_comp,
        with all the points that are in group group. """
        chemin = cut_last_slash(chemin_comp) + group + ".cor"
        endLine = " {} {} {}\n".format(sigX, sigY, sigZ)
        fixed_points,points = self.findPoints(group)
        with open(chemin, mode='w', encoding='utf-8') as fichier:
            for pt in fixed_points:
                line = "1 {} {} {} {}".format(pt['name'], pt['x'], pt['y'], pt['z']) + endLine
                fichier.write(line)
            fichier.write("\n")
            for pt in points:
                line = "0 {} {} {} {}".format(pt['name'], pt['x'], pt['y'], pt['z']) + endLine
                fichier.write(line)
                
    def create_obs_file(self, group, chemin_comp, sigma_hz, sigma_v, sigma_dist, sigma_dn):
        """ Create a .obs file with group as name and with path of chemin_comp,
        with all the observations that are in group group. """
        chemin = cut_last_slash(chemin_comp) + group + ".obs"
        endLine = " 0.0000   0.0000   0.0000 \n"
        niv_obs, tD_obs = self.findObs(group)
        with open(chemin, mode='w', encoding='utf-8') as fichier:
            for obs in niv_obs:
                line = "4 {} {} 0 {}".format(obs[0], obs[1], sigma_dn) + endLine
                fichier.write(line)
            fichier.write("\n")
            prems = True
            for obs in tD_obs:
                dep = obs[0]
                arr = obs[1]
                if prems:
                    line = "7 {} {} 0 {}".format(dep, arr, sigma_hz) + endLine
                else:
                    line = "5 {} {} 0 {}".format(dep, arr, sigma_hz) + endLine
                line+= "6 {} {} 0 {}".format(dep, arr, sigma_v) + endLine
                line+= "3 {} {} 0 {}".format(dep, arr, sigma_dist) + endLine + "\n"
                fichier.write(line)
                
    def create_comp_file(self, group, chemin_comp):
        chemin = cut_last_slash(chemin_comp) + group + ".comp"
        # get the crs of the group
        lay = QgsProject.instance().mapLayersByName('Stations - ' + group)[0]
        crss = lay.crs()
        ccrs = crss.toProj()
        # center coordinates = coords of 1st station pt
        stations = lay.getFeatures()
        c = {'x':0,'y':0,'z':0}
        for station in stations:
            c = self.getPoint(station)
            if c['z'] == None:
                c['z'] = 100
            break
        # write comp file
        with open(chemin, mode='w', encoding='utf-8') as fichier:
            fichier.write('data=\n{\n\t"config" : \n\t{\n\t\t"center_latitude" : 48.0,\n')
            fichier.write('\t\t"clean_outputs" : false,\n\t\t"compute_type" : 1,\n')
            fichier.write('\t\t"convergence_criterion" : 0.001,\n\t\t"coord_cov_file" : "",\n')
            fichier.write('\t\t"description" : "",\n\t\t"display_map" : true,\n')
            fichier.write('\t\t"files_unit" : 1,\n\t\t"force_iterations" : 0,\n')
            fichier.write('\t\t"internal_constraints" : false,\n\t\t"invert_matrix" : true,\n')
            fichier.write('\t\t"lang" : "fr",\n\t\t"local_center" : \n\t\t[\n')
            fichier.write('\t\t\t{},\n\t\t\t{},\n\t\t\t{}\n\t\t],\n'.format(c['x'], c['y'], c['z']))
            fichier.write('\t\t"max_iterations" : 100,\n\t\t"misc_metadata" : null,\n')
            fichier.write('\t\t"name" : "{}",\n'.format(group))
            fichier.write('\t\t"nb_digits" : 4,\n\t\t"proj_def" : "{}",\n'.format(ccrs))
            fichier.write('\t\t"refraction" : 0.12,\n')
            fichier.write('\t\t"root_COR_file" : "{}",\n'.format(group+".cor"))
            fichier.write('\t\t"root_OBS_file" : "{}",\n'.format(group+".obs"))
            fichier.write('\t\t"use_ellips_height" : true,\n\t\t"use_proj" : true\n\t},\n')
            fichier.write('\t"config_file" : "{}"\n'.format(chemin))
            fichier.write('}')


    def new_simulation(self):
        """ Creates a stations point layer, a targets point layer and an obs
        line layer, on which one can draw the simulation configuration """
        src = self.selecteur_de_src.crs().toProj()
        nom = self.nom_nlle_sim.text()
        self.create_4_layers_in_group(src,nom)
        self.populate_group_combobox()
        
    def populate_group_combobox(self):
        """ Done when "connecting signals" and when creating a new simulation"""
        groups = QgsProject.instance().layerTreeRoot().findGroups()
        names = []
        for group in groups:
            names.append(group.name())
        self.combo_nom_sim.clear()
        self.combo_nom_sim.insertItems(0,names)

    def generate_simulation(self):
        """ Generate a .comp file, a .cor file and a .obs file according to
        the drawings in the 3 simulation layers """
        group = self.combo_nom_sim.currentText()
        compFilePath = self.chemin_fichier_sim.filePath()
        sigma_hz = self.sigma_hz.value()
        sigma_v = self.sigma_v.value()
        sigma_dist = self.sigma_dist.value()
        sigma_dn = self.sigma_dn.value()
        sigma_x = self.sigma_x.value()
        sigma_y = self.sigma_y.value()
        sigma_z = self.sigma_z.value()
        
        self.create_cor_file(group, compFilePath, sigma_x, sigma_y, sigma_z)
        self.create_obs_file(group, compFilePath, sigma_hz, sigma_v, sigma_dist, sigma_dn)
        self.create_comp_file(group, compFilePath)


    def debug_controler(self):
        # self.gpkg_db.display_gpkg_debug()
        pass

    def showEvent(self, a0: QtGui.QShowEvent) -> None:
        self.connecting_signals()
       

    def closeEvent(self, a0: QtGui.QCloseEvent) -> None:
        self.disconnecting_signals()
        print("Antesim window closed !!!")
        
