from __future__ import absolute_import

import logging
import os
import sqlite3
import time
import random

from PyQt5.QtCore import QDate, QDateTime
from PyQt5.QtSql import QSqlDatabase
from osgeo import gdal, ogr
from qgis.core import QgsVectorLayer, QgsFields, QgsFeature, QgsPoint,QgsPointXY,  QgsVectorFileWriter, QgsWkbTypes, \
    QgsCoordinateTransformContext, QgsCoordinateReferenceSystem, QgsLineString, QgsDataSourceUri, QgsProject, \
    QgsVectorLayerUtils, edit, QgsAttributeTableConfig, QgsGeometry, QgsPointXY, QgsCategorizedSymbolRenderer, QgsFillSymbol, QgsRendererCategory
from PyQt5.QtWidgets import QMessageBox
from PyQt5.QtGui import QIcon, QColor

from .comp3d_computation import Comp3dComputation
from ..compgis_controler import CompGisControler
from qgis import processing
import numpy as np


class QgsDataSourceURI(object):
    pass


class GPKGManager:
    def __init__(self, controler: CompGisControler):
        self.controler = controler
        self.db_file = controler.gpkg_path


    def compute_already_exists(self, id_computation):
        """
            check if the computation is already loaded
        """
        computations_table = QgsVectorLayer(self.db_file + "|layername=computations", "computations", "ogr")
        lst_id, res = QgsVectorLayerUtils.getValues(computations_table, "timestamp_id")

        if int(id_computation) in lst_id:
            return True
        else:
            return False

    def relprec_already_exists(self, id_relprec):
        """
            check if the file relative precisions are already loaded
        """
        relprec_table = QgsVectorLayer(self.db_file + f"|layername=relprec_{id_relprec}", "précision relative", "ogr")
        lst_id, res = QgsVectorLayerUtils.getValues(relprec_table, "idcomp")

        if id_relprec in lst_id:
            return True
        else:
            return False

    def get_list_points(self, id_computation):
        """
            returns the list of all points in the computation (id_computation) 
        """
        lyr = self.get_layer_from_id_and_type(id_computation, 'pt')
        lst_pts, res = QgsVectorLayerUtils.getValues(lyr, "name")
        if res:
            return lst_pts

    def get_computation_info(self, id_computation):
        """
            returns informations on the computation (id_computation)
        """
        computations_table = QgsVectorLayer(self.db_file + "|layername=computations", "computations", "ogr")
        for feat in computations_table.getFeatures():
            if feat.id() == int(id_computation):
                return feat
        return None

    def get_computations_table(self):
        computations_table = QgsVectorLayer(self.db_file + "|layername=computations", "computations", "ogr")
        return computations_table

    def get_layer_from_id_and_type(self, id_computation: str, type_lyr: str):
        """
        :param id_computation: to be sure to catch only the layers of this computation
        :param type_lyr: match with the first part of the names of layers that are built by the plugin
        :return: layer or None
        """
        if type_lyr not in ['pt', 'obs', 'ell', 'relprec', 'kernel']:
            return None
        layer = QgsVectorLayer(self.db_file, "all", "ogr")
        subLayers = layer.dataProvider().subLayers()
        for subLayer in subLayers:
            try:
                name = subLayer.split('!!::!!')[1]
                t_lyr = name.split('_')[0]
                id_lyr = name.split('_')[1]
                if id_computation == id_lyr and t_lyr == type_lyr:
                    uri = "%s|layername=%s" % (self.db_file, name,)
                    # get llayer
                    sub_vlayer = QgsVectorLayer(uri, name, 'ogr')
                    return sub_vlayer
            except Exception:
                return None
        return None

    def delete_computation(self, id_computation):
        """
            Remove feature from computations table and associated layers
        """
        driver_name = "GPKG"
        drv = ogr.GetDriverByName(driver_name)
        gpkg_conn = drv.Open(self.db_file, 1)
        lyr_computation = gpkg_conn.GetLayerByName('computations')
        if lyr_computation is None:
            print("problème ! computations table not exists")
            return

        layerList = []
        for i in gpkg_conn:
            daLayer = i.GetName()
            if not daLayer in layerList:
                layerList.append(daLayer)

        layerList.sort()
        # Delete feature in Computations table
        for feature in lyr_computation:
            if str(feature.GetFID()) == id_computation:
                lyr_computation.DeleteFeature(feature.GetFID())

                try:
                    gpkg_conn.ExecuteSQL(f'DROP TABLE obs_{id_computation}')
                    gpkg_conn.ExecuteSQL(f'DROP TABLE pt_{id_computation}')
                    gpkg_conn.ExecuteSQL(f'DROP TABLE ell_{id_computation}')
                    gpkg_conn.ExecuteSQL(f'DROP TABLE relprec_{id_computation}')
                    gpkg_conn.ExecuteSQL(f'DROP TABLE kernel_{id_computation}')
                except Exception as e:
                    print("problem to delete layer", e)
        gpkg_conn.ExecuteSQL('VACUUM')
        gpkg_conn = None

    def add_compute_to_map(self, id_computation, group,invert_matrix):
        """
        If compute not in Qgis, add all layers associated to this computations in a group layer
        :param id_computation: timestamp of computation
        :return:
        """
        layer = QgsVectorLayer(self.db_file, "all", "ogr")
        subLayers = layer.dataProvider().subLayers()
                
        for subLayer in subLayers:
            name = subLayer.split('!!::!!')[1]  # Récupérer le nom complet
            
            if id_computation in name:
                uri = "%s|layername=%s" % (self.db_file, name)
                
                # Créer la couche
                sub_vlayer = QgsVectorLayer(uri, name, 'ogr')
                # Séparer le nom et le suffixe
                if "_" in name:
                    base_name, suffix = name.split("_", 1)
                    if base_name == "pt":
                        display_name = "points"
                    elif base_name == "obs":
                        display_name = "observations"
                    elif base_name == "ell":
                        display_name = "ellipses"
                    elif base_name == "relprec":
                        display_name = "précision relative"
                    elif base_name == "kernel":
                        display_name == "kernel"
                else:
                    display_name = name

                # Charger le style et réordonner les champs
                lyr_styled = self.load_style(sub_vlayer, invert_matrix)
                lyr_styled = self.reorder_fields(lyr_styled)

                # Modifier le nom affiché
                sub_vlayer.setName(display_name)  
        
                # Ajouter la couche au projet
                QgsProject.instance().addMapLayer(lyr_styled, False)
        
                # Ajouter la couche au groupe
                group.addLayer(lyr_styled)

        # changer le nom du group pour avoir la date de calcul en plus du nom.
        current_name = group.name()
        suffix = int(suffix)
        suffix = str(QDateTime().fromMSecsSinceEpoch(suffix).toString('yyyy-MM-dd hh:mm:ss'))
        name_group = f"{current_name} :\n      {suffix}"
        group.setName(name_group)
                
    def reorder_fields(self, lyr):
        """
        Reorder fields in attribute table
        :param lyr: layer to reorder
        :return: layer ordered
        """

        if 'obs' in lyr.name():
            config = lyr.attributeTableConfig()
            columns = config.columns()
            config.setColumns([self.getColumnIndex(config, 'active'),
                               self.getColumnIndex(config, 'code'),
                               self.getColumnIndex(config, 'from_pt'),
                               self.getColumnIndex(config, 'to'),
                               self.getColumnIndex(config, 'originale_value'),
                               self.getColumnIndex(config, 'computed_value'),
                               self.getColumnIndex(config, 'residual'),
                               self.getColumnIndex(config, 'normalized_residual'),
                               self.getColumnIndex(config, 'sigma_total'),
                               self.getColumnIndex(config, 'to_point')
                               ])
            lyr.setAttributeTableConfig(config)

        return lyr

    def getColumnIndex(self, config, key):
        """
        Find field's name and return index
        :param config: attributeTableConfig
        :param key: field to search
        :return: index
        """
        for i, col in enumerate(config.columns()):
            if col.name == key:
                return config.columns()[i]

        # If not found
        c = QgsAttributeTableConfig.ColumnConfig()
        c.name = key
        return c

    def save_comp_to_gpkg(self, comp: Comp3dComputation):

        # Nouveau calcul
        # On créé le gpkg s'il n'existe pas, et on ajoute une table computations
        if not os.path.exists(self.db_file):
            lyr_computation = comp.get_layer_dfn()
            save_options = QgsVectorFileWriter.SaveVectorOptions()
            save_options.layerName = 'computations'
            save_options.layerOptions = ['FID=timestamp_id']
            transform_context = QgsProject.instance().transformContext()
            save_options.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteFile
            error = QgsVectorFileWriter.writeAsVectorFormatV2(lyr_computation,
                                                              self.db_file,
                                                              transform_context,
                                                              save_options
                                                              )
            if error[0] == QgsVectorFileWriter.NoError:
                pass
            else:
                MB = QMessageBox()
                MB.setWindowTitle("ERREUR!")
                MB.setText(f"il y a eu un problème à la création du Géopackage\n {error}")
                MB.setWindowIcon(QIcon(":/plugins/Compgis3/resources/CompGIS.png"))
                MB.exec()

        # On ajoute une feature comp dans la table
        # Puis ajout des lyers pts, obs, et éventuellement ellipses

        computations_table = QgsVectorLayer(self.db_file + "|layername=computations", "computations", "ogr")
        if computations_table.isValid():
            pr = computations_table.dataProvider()
            lst_values = comp.get_as_list_fields_values()
            fet = QgsFeature()
            fet.setAttributes(lst_values)
            (res, feat) = pr.addFeatures([fet])
            fet = None

            if res:
                self.save_lyr_pts_to_gpkg(comp)
                self.save_lyr_obs_to_gpkg(comp)
                if comp.lst_indeterminations != []:
                    self.save_lyr_krnl_to_gpkg(comp)
                if comp.invert_matrix == 1:
                    self.save_lyr_ell_to_gpkg(comp)
                return True

    def save_lyr_pts_to_gpkg(self, comp: Comp3dComputation):
        lyr = QgsVectorLayer('Point', 'table_name', 'memory')
        # lyr.startEditing()
        lyr_provider = lyr.dataProvider()
        lyr_name = f'pt_{comp.timestamp_id}'

        for i, pt in enumerate(comp.lst_points):
            if i == 0:
                _fields = QgsFields()
                for field in pt.get_qgis_fields():
                    _fields.append(field)
                error = lyr.dataProvider().addAttributes(_fields)
                lyr.updateFields()
            lyr.commitChanges()

            feat = QgsFeature(_fields)
            for attr in pt.get_attributes():
                val = getattr(pt, attr)
                if dict == type(val):
                    feat[attr] = str(val)
                elif list == type(val):
                    feat[attr] = str(val)
                else:
                    feat[attr] = val

            ptz = QgsPoint(pt.get_x(), pt.get_y(), pt.get_h())
            feat.setGeometry(ptz)

            (res, outFeats) = lyr_provider.addFeatures([feat])
            if not res:
                print("pb à l'ajout de la feature", outFeats.attributes())


        # Write to gpkg
        opts = QgsVectorFileWriter.SaveVectorOptions()
        self.write_to_gpkg(opts, lyr,  lyr_name, comp, QgsWkbTypes.PointZ)

    def save_lyr_ell_to_gpkg(self, comp: Comp3dComputation):
        lyr = QgsVectorLayer('Point', 'table_name', 'memory')
        # lyr.startEditing()
        lyr_provider = lyr.dataProvider()
        lyr_name = f'ell_{comp.timestamp_id}'

        for i, ell in enumerate(comp.lst_ellipses):
            if i == 0:
                fields = QgsFields()

                for field in ell.get_qgis_fields():
                    fields.append(field)
                error = lyr.dataProvider().addAttributes(fields)

                lyr.updateFields()
            lyr.commitChanges()

            feat = QgsFeature(fields)
            for attr in ell.get_attributes():
                val = getattr(ell, attr)
                if dict == type(val):
                    feat[attr] = str(val)
                elif list == type(val):
                    feat[attr] = str(val)
                elif np.ndarray == type(val):
                    feat[attr] = str(val)
                elif float == type(val):
                    feat[attr] = float(val)
                else:
                    feat[attr] = val

            pt = comp.get_pt_from_name(ell.pt_name)
            ptz = QgsPoint(pt.get_x(), pt.get_y(), pt.get_h())

            feat.setGeometry(ptz)

            (res, outFeats) = lyr_provider.addFeatures([feat])
            if not res:
                print("pb à l'ajout de la feature", outFeats)


            # Write to gpkg
        opts = QgsVectorFileWriter.SaveVectorOptions()
        self.write_to_gpkg(opts, lyr,  lyr_name, comp, QgsWkbTypes.PointZ)

    def save_lyr_obs_to_gpkg(self, comp: Comp3dComputation):
        lyr = QgsVectorLayer('Linestring', 'table_name', 'memory')
        lyr_provider = lyr.dataProvider()
        lyr_name = f'obs_{comp.timestamp_id}'
        fields = QgsFields()
        for i, lin in enumerate(comp.lst_observations):
            for field in lin.get_qgis_fields():
                if not field in fields:
                    fields.append(field)

        error = lyr.dataProvider().addAttributes(fields)
        lyr.updateFields()
        lyr.commitChanges()

        for i, lin in enumerate(comp.lst_observations):
            feat = QgsFeature(fields)
            for attr in lin.get_attributes():
                val = getattr(lin, attr)
                try:
                    if dict == type(val):
                        feat[attr] = str(val)
                    elif list == type(val):
                        feat[attr] = str(val)
                    else:
                        feat[attr] = val
                except KeyError as ke:
                    pass

            pt_from_name = lin.from_pt
            pt_to_name = lin.to
            pt = comp.get_pt_from_name(pt_from_name)
            ptz_from = QgsPoint(pt.get_x(), pt.get_y(), pt.get_h())
            pt = comp.get_pt_from_name(pt_to_name)
            ptz_to = QgsPoint(pt.get_x(), pt.get_y(), pt.get_h())
            lineZ = QgsLineString(ptz_from, ptz_to)
            feat.setGeometry(lineZ)

            (res, outFeats) = lyr_provider.addFeatures([feat])
            if not res:
                print("pb à l'ajout de la feature", outFeats.attributes())


        # Write to gpkg
        opts = QgsVectorFileWriter.SaveVectorOptions()
        self.write_to_gpkg(opts, lyr,  lyr_name, comp, QgsWkbTypes.LineStringZ)

    def save_lyr_krnl_to_gpkg(self, comp: Comp3dComputation):
        lyr = QgsVectorLayer('Polygon', 'table_name', 'memory')
        lyr_provider = lyr.dataProvider()
        lyr_name = f'kernel_{comp.timestamp_id}'

        fields = QgsFields()
        for kn in comp.lst_indeterminations:
            for field in kn.get_qgis_fields():
                if not field in fields:
                    fields.append(field)
 
        error = lyr.dataProvider().addAttributes(fields)
        lyr.updateFields()
        lyr.commitChanges()

        for kn in comp.lst_indeterminations:
            feat = QgsFeature(fields)
            
            for attr in kn.get_attributes():
                val = getattr(kn, attr)

                if dict == type(val):
                    feat[attr] = str(val)
                elif list == type(val):
                    feat[attr] = str(val)
                else:
                    feat[attr] = val
 
            flat_lst_coord = [coord for sublist in kn.lst_coord for coord in sublist]
            collection_point = [QgsPointXY(x, y) for x, y in flat_lst_coord]
            all_x_values = [x for x, y in flat_lst_coord]
            all_y_values = [y for x, y in flat_lst_coord]

            if len(collection_point) > 3:  
                collection_point.append(collection_point[0])
                polygon = QgsGeometry.fromPolygonXY([collection_point])
                convex_hull = polygon.convexHull()
                feat.setGeometry(convex_hull)
            
            
            elif len(collection_point) in [2,3] :
                xmin, xmax = min(all_x_values), max(all_x_values)
                ymin, ymax = min(all_y_values), max(all_y_values)
                kn1 = QgsPointXY(xmin,ymin)
                kn2 = QgsPointXY(xmin,ymax)
                kn3 = QgsPointXY(xmax,ymax)
                kn4 = QgsPointXY(xmax,ymin)
                polygon = QgsGeometry.fromPolygonXY([[kn1,kn2,kn3,kn4,kn1]])
                feat.setGeometry(polygon)
    
            elif len(collection_point) == 1:
                x = min(all_x_values)
                y = min(all_y_values)
                kn1 = QgsPointXY(x-1,y-1)
                kn2 = QgsPointXY(x-1,y+1)
                kn3 = QgsPointXY(x+1,y+1)
                kn4 = QgsPointXY(x+1,y-1)
                polygon = QgsGeometry.fromPolygonXY([[kn1,kn2,kn3,kn4,kn1]])
                feat.setGeometry(polygon)

            (res, outFeats) = lyr_provider.addFeatures([feat])
            if not res:
                print("pb à l'ajout de la feature", outFeats[0].attributes())


        # Write to gpkg
        opts = QgsVectorFileWriter.SaveVectorOptions()
        self.write_to_gpkg(opts, lyr,  lyr_name, comp, QgsWkbTypes.Polygon)

    def save_lyr_relprec_to_gpkg(self, relprec):
        lyr = QgsVectorLayer('Point', 'table_name', 'memory')
        lyr_provider = lyr.dataProvider()
        lyr_name = f'relprec_{relprec.active_compute}'

        for i, pt in enumerate(relprec.lst_pt_relprec):
            if i == 0:
                _fields = QgsFields()
                for field in pt.get_qgis_fields():
                    _fields.append(field)
                error = lyr.dataProvider().addAttributes(_fields)
                lyr.updateFields()
            lyr.commitChanges()

            feat = QgsFeature(_fields)
            for attr in pt.get_attributes():
                val = getattr(pt, attr)
                if dict == type(val):
                    feat[attr] = str(val)
                elif list == type(val):
                    feat[attr] = str(val)
                else:
                    feat[attr] = val
            ptz = QgsPoint(float(pt.x), float(pt.y), float(pt.z))
            feat.setGeometry(ptz)

            (res, outFeats) = lyr_provider.addFeatures([feat])
            if not res:
                print("pb à l'ajout de la feature", outFeats.attributes())


        # Write to gpkg
        opts = QgsVectorFileWriter.SaveVectorOptions()
        # self.write_to_gpkg(opts, lyr,  lyr_name, comp, 'PointZ')
        if not os.path.exists(self.db_file):
            opts.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteFile
        else:
            opts.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteLayer
        
        opts.driverName = "GPKG"
        opts.overrideGeometryType = QgsWkbTypes.PointZ
        context = QgsCoordinateTransformContext()
        opts.layerName = lyr_name
        crs = relprec.crs
        lyr.setCrs(crs)
        
        error = QgsVectorFileWriter.writeAsVectorFormatV2(layer=lyr,
                                                          fileName=self.db_file,
                                                          transformContext=context,
                                                          options=opts,
                                                          )
        
        if error[0] == QgsVectorFileWriter.NoError:
            pass
        else:
            MB = QMessageBox()
            MB.setWindowTitle("ERREUR!")
            MB.setText(f"il y a eu un problème à la création de la couche pt\n {error}")
            MB.setWindowIcon(QIcon(":/plugins/Compgis3/resources/CompGIS.png"))
            MB.exec()
        lyr = None

    def write_to_gpkg(self, opts,lyr, lyr_name, comp, geom_type):
        if not os.path.exists(self.db_file):
            opts.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteFile
        else:
            opts.actionOnExistingFile = QgsVectorFileWriter.CreateOrOverwriteLayer
        
        opts.driverName = "GPKG"
        opts.overrideGeometryType = geom_type
        context = QgsCoordinateTransformContext()
        opts.layerName = lyr_name
        crs = self.create_crs(comp, lyr)
        lyr.setCrs(crs)
        
        error = QgsVectorFileWriter.writeAsVectorFormatV2(layer=lyr,
                                                          fileName=self.db_file,
                                                          transformContext=context,
                                                          options=opts,
                                                          )
        
        if error[0] == QgsVectorFileWriter.NoError:
            pass
        else:
            MB = QMessageBox()
            MB.setWindowTitle("ERREUR!")
            MB.setText(f"il y a eu un problème à la création de la couche pt\n {error}")
            MB.setWindowIcon(QIcon(":/plugins/Compgis3/resources/CompGIS.png"))
            MB.exec()
        lyr = None
    
    def create_crs(self, comp, lyr):
        """
        Create crs from computation parameters
        :param lyr:
        :param comp:
        :return: crs
        """
        crs = lyr.crs()
        if comp.input_proj_EPSG != -1:
            code = int(comp.input_proj_EPSG)
            crs.createFromId(code)
        else:
            crs = QgsCoordinateReferenceSystem()
            crs.createFromProj4(comp.input_proj_def)
        return crs
        
    def load_style(self, lyr,invert_matrix):
        """
            param invert_matrix: use to check if layer ellipse is present.
            load default_style saved for all kinds of layers but for 'kernel'.
            it's style is made here to be abble to be dynamic
        """
        
        layer_type = lyr.name().split('_')[0]
        
        if layer_type == 'pt':
            path_style = os.path.join(self.controler.plugin_dir, 'default_style', 'style_points.qml')
            lyr.loadNamedStyle(path_style)
        elif layer_type == 'ell':
            path_style = os.path.join(self.controler.plugin_dir, 'default_style', 'style_ell.qml')
            lyr.loadNamedStyle(path_style)
        elif layer_type == 'obs':
            if invert_matrix:
                path_style = os.path.join(self.controler.plugin_dir, 'default_style', 'style_obs_inverted.qml')
            else:
                path_style = os.path.join(self.controler.plugin_dir, 'default_style', 'style_obs.qml')
            lyr.loadNamedStyle(path_style)
        elif layer_type == "relprec":
            path_style = os.path.join(self.controler.plugin_dir, 'default_style', 'style_relprec.qml')
            lyr.loadNamedStyle(path_style)
        elif layer_type == "kernel":

            # construction d'un style dynamique dépendant du nombre de noyaux.

            attribute_name = "indetermination"
            categories = []
            values = [feature[attribute_name] for feature in lyr.getFeatures()]
            
            for i, value in enumerate(values):
                # création d'une couleur en teinte saturation valeur
                hue = (i/len(values))
                color = QColor.fromHsvF(hue,1.0,1.0)
                # "allègement" de la couleur
                color.lighter(150)
                symbol = QgsFillSymbol.createSimple({'color':color.name()})
                # règle l'opacité à 30%
                symbol.setOpacity(0.3)
                category = QgsRendererCategory(value, symbol, str(value))
                categories.append(category)
            # filtre à partir du champ indetermination
            renderer = QgsCategorizedSymbolRenderer(attribute_name, categories)
            lyr.setRenderer(renderer)

            lyr.triggerRepaint()

        return lyr
        
    def display_gpkg_debug(self):
        print('***** DEBUG GPKG ********************')
        driver_name = "GPKG"
        drv = ogr.GetDriverByName(driver_name)
        gpkg_conn = drv.Open(self.db_file, 0)
        for layer in gpkg_conn:
            print('Layer Name:', layer.GetName())
            print('Layer Feature Count:', len(layer))
            # each layer has a schema telling us what fields and geometric fields the features contain
            print('Layer Schema')
            layer_defn = layer.GetLayerDefn()
            for i in range(layer_defn.GetFieldCount()):
                print(layer_defn.GetFieldDefn(i).GetName())
            # some layers have multiple geometric feature types
            # most of the time, it should only have one though
            for i in range(layer_defn.GetGeomFieldCount()):
                # some times the name doesn't appear
                # but the type codes are well defined
                print(layer_defn.GetGeomFieldDefn(i).GetName(), layer_defn.GetGeomFieldDefn(i).GetType())
            # get a feature with GetFeature(featureindex)
            # this is the one where featureindex may not start at 0
            layer.ResetReading()
            for feature in layer:
                print('Feature ID:', feature.GetFID())
                # get a metadata field with GetField('fieldname'/fieldindex)
                print('Feature Metadata Keys:', feature.keys())
                print('Feature Metadata Dict:', feature.items())
                print('Feature Geometry:', feature.geometry())

        gpkg_conn = None


